# This script is part 3 of the pipeline; It:
# 1. performs NEE gap-filling 
# 2. exports results as df_openeddy_SSH_gapfill.csv 
# 
# Felix Nieberding
# 2020-08-26


rm(list=ls())
Sys.setenv(TZ='UTC')

library(tidyverse)
library(lubridate)
library(data.table)
library(REddyProc)
library(lognorm)
library(pracma)

# select the respective dataset
dataset <- "ORG"
# dataset <- "WLG"

setwd(dir = "~/TransTiP/_NamCo_sync/3_data_qa-qc/Flux data/ESSD_scripts_review/")

# Import data -------------------------------------------------------------
# import openeddy_Burba data with drift correction
df <- fread(paste("df_", dataset, "_openeddy_SSH.csv", sep = "")) %>% mutate(DATETIME = as.POSIXct(DATETIME))

# REddyProc ---------------------------------------------------------------
EddyData <- df %>%
  transmute(DateTime = DATETIME,
            daytime = daytime,
            NEE = F_CO2, 
            Rg = ifelse(is.na(SWin)== T & is.na(Rg) == F, Rg, SWin), # (W/m²) fill missing SWin with Rg,
            Tair = Ta - 273.15,
            Tsoil = Ts - 273.15,
            rH = RH.y,
            VPD = fCalcVPDfromRHandTair(rH = rH, Tair = Tair),
            Ustar = u.) %>%
  filterLongRuns("NEE")

# Exclude night-time date < -0.1 and remove upper and lower 0.2 percent
EddyData <- EddyData %>%
  mutate(NEE = ifelse(daytime == 0 & NEE < -0.1, NA, NEE), 
         NEE = ifelse(NEE < quantile(NEE, probs = 0.002, na.rm = T) | NEE > quantile(NEE, probs = 0.998, na.rm = T), NA, NEE)) %>%
  select(-daytime)

# Initalize R5 reference class sEddyProc for post-processing of eddy data
EProc <- sEddyProc$new(paste("ECNM_", dataset, sep = ""), EddyData, 
                       c('NEE', 'Rg', 'Tair', 'VPD', 'Ustar', 'Tsoil', "rH"))

# fingerprint plot before gapfilling
EProc$sPlotFingerprint('NEE', Dir = getwd())

# gap filling without ustar filtering
EProc$sMDSGapFill('NEE')

# fingerprint plot after gapfilling
EProc$sPlotFingerprint('NEE_f', Dir = getwd())

# export results ----------------------------------------------------------
FilledEddyData <- EProc$sExportResults()
CombinedData <- cbind(EddyData, FilledEddyData)

CombinedData %>%
  mutate(DATETIME = as.character(DateTime)) %>%
  fwrite(paste("df_", dataset, "_openeddy_SSH_gapfill.csv", sep = ""), na = "-9999")

